package com.example.a66text

/*
  SmsReceiver.kt
  BroadcastReceiver to listen for incoming SMS messages and forward them to the PHP API endpoint if the app is authenticated.
*/

/*
  Listens for the SMS_RECEIVED broadcast and posts SMS details to the API after login.
*/
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.telephony.SmsMessage
import android.util.Log
import okhttp3.*
import java.io.IOException

class SmsReceiver : BroadcastReceiver() {

    /*
      Called automatically when an SMS is received.
      Checks for valid API configuration and forwards SMS if authenticated.
    */
    override fun onReceive(context: Context, intent: Intent) {
        if (intent.action == "android.provider.Telephony.SMS_RECEIVED") {

            /* Get the saved API config from SharedPreferences */
            val shared_preferences = context.getSharedPreferences("app_prefs", Context.MODE_PRIVATE)
            val api_key = shared_preferences.getString("pref_api_key", "")
            val site_url = shared_preferences.getString("pref_site_url", "")
            val device_id = shared_preferences.getString("pref_device_id", "")

            /* Abort if API config is missing (not logged in) */
            if (api_key.isNullOrEmpty() || site_url.isNullOrEmpty() || device_id.isNullOrEmpty()) {
                Log.d("66text", "No API config, will not send incoming SMS to server")
                return
            }

            /* Extract SMS messages from the broadcast intent */
            val bundle: Bundle? = intent.extras
            if (bundle != null) {
                try {
                    val pdus = bundle.get("pdus") as Array<*>
                    val sms_messages = mutableListOf<SmsMessage>()

                    /* Collect all message parts */
                    for (pdu in pdus) {
                        sms_messages.add(SmsMessage.createFromPdu(pdu as ByteArray))
                    }

                    if (sms_messages.isNotEmpty()) {
                        /* Sort parts by sequence to keep correct order */
                        sms_messages.sortBy { it.indexOnIcc }

                        val phone_number = sms_messages[0].displayOriginatingAddress

                        /* Merge all parts into a single string */
                        val full_message = StringBuilder()
                        for (message_part in sms_messages) {
                            full_message.append(message_part.messageBody)
                        }

                        /* Get SIM subscription id (API 19+) */
                        val subscription_id = intent.extras?.getInt("subscription", -1) ?: -1

                        /* Send the complete merged message to the API */
                        send_sms_to_api(
                            context,
                            site_url,
                            api_key,
                            device_id,
                            phone_number,
                            full_message.toString(),
                            subscription_id
                        )
                    }
                } catch (exception: Exception) {
                    Log.e("66text", "SMS parsing failed: ${exception.message}")
                }
            }
        }
    }

    /*
      Sends the SMS data to the PHP API endpoint using OkHttp.
    */
    private fun send_sms_to_api(context: Context, site_url: String, api_key: String, device_id: String, phone_number: String, content: String, subscription_id: Int) {

        /* Get battery info */
        val battery_manager = context.getSystemService(Context.BATTERY_SERVICE) as android.os.BatteryManager
        val device_battery = battery_manager.getIntProperty(android.os.BatteryManager.BATTERY_PROPERTY_CAPACITY)
        val battery_status_intent = context.registerReceiver(null, android.content.IntentFilter(android.content.Intent.ACTION_BATTERY_CHANGED))
        val device_is_charging = if (battery_status_intent?.getIntExtra(android.os.BatteryManager.EXTRA_PLUGGED, -1) != 0) 1 else 0

        /* Build the URL for the API endpoint */
        val url = "${site_url}api/sms/receive"
        val client = OkHttpClient()

        /* Build the form data for the POST request */
        val form_body = FormBody.Builder()
            .add("device_id", device_id)
            .add("phone_number", phone_number)
            .add("content", content)
            .add("sim_subscription_id", subscription_id.toString())
            .add("device_battery", device_battery.toString())
            .add("device_is_charging", device_is_charging.toString())
            .build()

        /* Build and send the POST request asynchronously */
        val request = Request.Builder()
            .url(url)
            .addHeader("Authorization", "Bearer $api_key")
            .post(form_body)
            .build()

        client.newCall(request).enqueue(object : Callback {

            /* Handle failure to send */
            override fun onFailure(call: Call, exception: IOException) {
                Log.e("66text", "Failed to send SMS to API: ${exception.message}")
            }

            /* Handle successful response */
            override fun onResponse(call: Call, response: Response) {
                Log.d("66text", "Sent incoming SMS to API, code: ${response.code}")
                response.close()
            }
        })
    }
}